   /*******************************************************/
   /*      "C" Language Integrated Production System      */
   /*                                                     */
   /*             CLIPS Version 6.30  10/19/06            */
   /*                                                     */
   /*                 RETE UTILITY MODULE                 */
   /*******************************************************/

/*************************************************************/
/* Purpose: Provides a set of utility functions useful to    */
/*   other modules.                                          */
/*                                                           */
/* Principal Programmer(s):                                  */
/*      Gary D. Riley                                        */
/*                                                           */
/* Contributing Programmer(s):                               */
/*                                                           */
/* Revision History:                                         */
/*                                                           */
/*      6.24: Removed INCREMENTAL_RESET compilation flag.    */
/*                                                           */
/*            Rule with exists CE has incorrect activation.  */
/*            DR0867                                         */
/*                                                           */
/*      6.30: Added support for hashed alpha memories.       */
/*                                                           */
/*            Removed pseudo-facts used in not CEs.          */
/*                                                           */
/*************************************************************/

#define _RETEUTIL_SOURCE_

#include <stdio.h>
#define _STDIO_INCLUDED_

#include "setup.h"

#if DEFRULE_CONSTRUCT

#include "drive.h"
#include "engine.h"
#include "envrnmnt.h"
#include "incrrset.h"
#include "match.h"
#include "memalloc.h"
#include "moduldef.h"
#include "pattern.h"
#include "retract.h"
#include "router.h"

#include "reteutil.h"

/***************************************/
/* LOCAL INTERNAL FUNCTION DEFINITIONS */
/***************************************/

   static void                        TraceErrorToRuleDriver(void *,struct joinNode *,char *);
   static struct alphaMemoryHash     *FindAlphaMemory(void *,struct patternNodeHeader *,unsigned long);
   static unsigned long               AlphaMemoryHashValue(void *,struct patternNodeHeader *,unsigned long);
   static void                        UnlinkAlphaMemory(void *,struct patternNodeHeader *,struct alphaMemoryHash *);
   static void                        UnlinkAlphaMemoryBucketSiblings(void *,struct alphaMemoryHash *);
   static void                        InitializePMLinks(struct partialMatch *);
   static void                        UnlinkBetaPartialMatchfromAlphaAndBetaLineage(struct partialMatch *);
   
/***********************************************************/
/* PrintPartialMatch: Prints out the list of fact indices  */
/*   and/or instance names associated with a partial match */
/*   or rule instantiation.                                */
/***********************************************************/
globle void PrintPartialMatch(
  void *theEnv,
  char *logicalName,
  struct partialMatch *list)
  {
   struct patternEntity *matchingItem;
   short int i;

   for (i = 0; i < (int) list->bcount;)
     {
      if ((get_nth_pm_match(list,i) != NULL) &&
          (get_nth_pm_match(list,i)->matchingItem != NULL))
        {
         matchingItem = get_nth_pm_match(list,i)->matchingItem;
         if (matchingItem != NULL) (*matchingItem->theInfo->base.shortPrintFunction)(theEnv,logicalName,matchingItem);
        }
      i++;
      if (i < (int) list->bcount) EnvPrintRouter(theEnv,logicalName,",");
     }
  }

/**********************************************/
/* CopyPartialMatch:  Copies a partial match. */
/**********************************************/
globle struct partialMatch *CopyPartialMatch(
  void *theEnv,
  struct partialMatch *list,
  int addActivationSlot,
  int addDependencySlot)
  {
   struct partialMatch *linker;
   short int i;

   linker = get_var_struct(theEnv,partialMatch,sizeof(struct genericMatch) *
                                        (list->bcount + addActivationSlot + addDependencySlot - 1));

   InitializePMLinks(linker);
   linker->betaMemory = TRUE;
   linker->busy = FALSE;
   linker->activationf = addActivationSlot;
   linker->dependentsf = addDependencySlot;
   linker->notOriginf = FALSE;
   linker->counterf = FALSE;
   linker->bcount = list->bcount;
   linker->hashValue = 0;
   linker->hashExpr = NULL;

   for (i = 0; i < (int) linker->bcount; i++) linker->binds[i] = list->binds[i];

   if (addActivationSlot) linker->binds[i++].gm.theValue = NULL;
   if (addDependencySlot) linker->binds[i].gm.theValue = NULL;

   return(linker);
  }

/****************************************************/
/* InitializePMLinks: */
/****************************************************/
static void InitializePMLinks(
  struct partialMatch *theMatch)
  {
   theMatch->nextInNode = NULL;
   theMatch->prevInNode = NULL;
   theMatch->nextInRight = NULL;
   theMatch->prevInRight = NULL;
   theMatch->nextInBeta = NULL;
   theMatch->prevInBeta = NULL;
   theMatch->children = NULL;
   theMatch->rightParent = NULL;
   theMatch->leftParent = NULL;
  }
  
/***********************************************************/
/* UpdatePMLinks: . */
/***********************************************************/
globle void UpdatePMLinks(
  struct partialMatch *thePM,
  struct partialMatch *lhsBinds,
  struct partialMatch *rhsBinds,
  struct joinNode *join,
  intBool notBeta)
  {
   /*================================*/
   /* Update the node's linked list. */
   /*================================*/

   if (notBeta)
     {
      thePM->nextInNode = join->notBeta;
      if (join->notBeta != NULL)
        { join->notBeta->prevInNode = thePM; }
      join->notBeta = thePM;
     }
   else
     {
      thePM->nextInNode = join->beta;
      if (join->beta != NULL)
        { join->beta->prevInNode = thePM; }
      join->beta = thePM;
     }
   
   thePM->owner = join;

   /*======================================*/
   /* Update the alpha memory linked list. */
   /*======================================*/
      
   if (rhsBinds != NULL)
     {
      thePM->nextInRight = rhsBinds->children;
      if (rhsBinds->children != NULL)
        { rhsBinds->children->prevInRight = thePM; }
      rhsBinds->children = thePM;
      thePM->rightParent = rhsBinds;
    }
      
   /*=====================================*/
   /* Update the beta memory linked list. */
   /*=====================================*/

   if (lhsBinds != NULL)
     {
      thePM->nextInBeta = lhsBinds->children;
      if (lhsBinds->children != NULL)
        { lhsBinds->children->prevInBeta = thePM; }
      lhsBinds->children = thePM;
      thePM->leftParent = lhsBinds;
     }
  }

/***********************************************************/
/* UpdatePMAlphaLinks: . */
/***********************************************************/
globle void UpdatePMAlphaLinks(
  struct partialMatch *thePM,
  struct partialMatch *rhsBinds)
  {
   if (rhsBinds == NULL) return;
   
   /*
   if (rhsBinds->betaMemory)
     {
      thePM->nextInBeta = rhsBinds->children;
      if (rhsBinds->children != NULL)
        { rhsBinds->children->prevInBeta = thePM; }
      rhsBinds->children = thePM;
      thePM->rightParent = rhsBinds; 
      return;
     }
*/

   thePM->nextInRight = rhsBinds->children;
   if (rhsBinds->children != NULL)
     { rhsBinds->children->prevInRight = thePM; }
   rhsBinds->children = thePM;
   thePM->rightParent = rhsBinds;
  }
  
/***********************************************************/
/* UnlinkBetaPMFromNode: . */
/***********************************************************/
globle void UnlinkBetaPMFromNode(
  struct joinNode *join,
  struct partialMatch *thePM,
  intBool notBeta)
  {
   /*=========================================*/
   /* Update the nextInNode/prevInNode links. */
   /*=========================================*/
   
   if (thePM->prevInNode == NULL)
     { 
      if (join != NULL)
        { 
         if (notBeta)
           { join->notBeta = thePM->nextInNode; }
         else
           { join->beta = thePM->nextInNode; }
        }
     }
   else
     { thePM->prevInNode->nextInNode = thePM->nextInNode; }

   if (thePM->nextInNode != NULL)
     { thePM->nextInNode->prevInNode = thePM->prevInNode; }
     
   thePM->nextInNode = NULL;
   thePM->prevInNode = NULL;
  } 
  
/***********************************************************/
/* LinkBetaPMToNode: . */
/***********************************************************/
globle void LinkBetaPMToNode(
  struct joinNode *join,
  struct partialMatch *thePM,
  intBool notBeta)
  {
   if (notBeta)
     {
      thePM->nextInNode = join->notBeta;
      if (join->notBeta != NULL)
        { join->notBeta->prevInNode = thePM; }
      join->notBeta = thePM;
     }
   else
     {
      thePM->nextInNode = join->beta;
      if (join->beta != NULL)
        { join->beta->prevInNode = thePM; }
      join->beta = thePM;
     }
  }

/***********************************************************/
/* UnlinkBetaPMFromNodeAndLineage: . */
/***********************************************************/
globle void UnlinkBetaPMFromNodeAndLineage(
  struct joinNode *join,
  struct partialMatch *thePM,
  intBool notBeta)
  {
   /*=========================================*/
   /* Update the nextInNode/prevInNode links. */
   /*=========================================*/
   
   if (thePM->prevInNode == NULL)
     { 
      if (join != NULL)
        { 
         if (notBeta)
           { join->notBeta = thePM->nextInNode; }
         else
           { join->beta = thePM->nextInNode; }
        }
     }
   else
     { thePM->prevInNode->nextInNode = thePM->nextInNode; }

   if (thePM->nextInNode != NULL)
     { thePM->nextInNode->prevInNode = thePM->prevInNode; }
     
   thePM->nextInNode = NULL;
   thePM->prevInNode = NULL;
   
   UnlinkBetaPartialMatchfromAlphaAndBetaLineage(thePM);
  } 

/*******************************************************************/
/* UnlinkBetaPartialMatchfromAlphaAndBetaLineage: Removes the      */
/*   lineage links from a beta memory partial match. This removes  */
/*   the links between this partial match and its left and right   */
/*   memory parents. It also removes the links between this        */
/*   partial match and any of its children in other beta memories. */
/*******************************************************************/
static void UnlinkBetaPartialMatchfromAlphaAndBetaLineage(
  struct partialMatch *thePM)
  {
   struct partialMatch *tempPM;
   
   /*=========================*/
   /* Update the alpha lists. */
   /*=========================*/

   if (thePM->prevInRight == NULL)
     { 
      if (thePM->rightParent != NULL)
        { 
         thePM->rightParent->children = thePM->nextInRight; 
         if (thePM->nextInRight != NULL)
           {
            thePM->rightParent->children = thePM->nextInRight;
            thePM->nextInRight->rightParent = thePM->rightParent;
           }
        } 
     }
   else
     { thePM->prevInRight->nextInRight = thePM->nextInRight; }

   if (thePM->nextInRight != NULL)
     { thePM->nextInRight->prevInRight = thePM->prevInRight; }

   thePM->rightParent = NULL;
   thePM->nextInRight = NULL;
   thePM->prevInRight = NULL;
   
   /*========================*/
   /* Update the beta lists. */
   /*========================*/

   if (thePM->prevInBeta == NULL)
     { 
      if (thePM->leftParent != NULL)
        { 
         thePM->leftParent->children = thePM->nextInBeta;
         if (thePM->nextInBeta != NULL)
           {
            thePM->leftParent->children = thePM->nextInBeta;
            thePM->nextInBeta->leftParent = thePM->leftParent;
           }
        } 
     }
   else
     { thePM->prevInBeta->nextInBeta = thePM->nextInBeta; }

   if (thePM->nextInBeta != NULL)
     { thePM->nextInBeta->prevInBeta = thePM->prevInBeta; }

   thePM->leftParent = NULL;
   thePM->nextInBeta = NULL;
   thePM->prevInBeta = NULL;
   
   /*===============================================*/
   /* Remove parent reference from the child links. */
   /* TBD just the first link.                      */
   /*===============================================*/
   
   if (thePM->children != NULL)
     {
      if (thePM->betaMemory)
        {
         for (tempPM = thePM->children; tempPM != NULL; tempPM = tempPM->nextInBeta)
           { tempPM->leftParent = NULL; }
        }
      else
        {
         for (tempPM = thePM->children; tempPM != NULL; tempPM = tempPM->nextInRight)
           { tempPM->rightParent = NULL; }
        }
        
      thePM->children = NULL;
     }
  } 

/*******************************************************************/
/* UnlinkBetaPMFromRight: TBD Change to RightMemory rather than alpha. */
/*******************************************************************/
globle void UnlinkBetaPMFromRight(
  struct partialMatch *thePM)
  {
   if (thePM->prevInRight == NULL)
     { 
      if (thePM->rightParent != NULL)
        { 
         thePM->rightParent->children = thePM->nextInRight; 
         if (thePM->nextInRight != NULL)
           {
            thePM->rightParent->children = thePM->nextInRight;
            thePM->nextInRight->rightParent = thePM->rightParent;
           }
        } 
     }
   else
     { thePM->prevInRight->nextInRight = thePM->nextInRight; }

   if (thePM->nextInRight != NULL)
     { thePM->nextInRight->prevInRight = thePM->prevInRight; }

   thePM->nextInRight = NULL;
   thePM->prevInRight = NULL;

   thePM->rightParent = NULL;
  }
  
/****************************************************/
/* MergePartialMatches: Merges two partial matches. */
/****************************************************/
globle struct partialMatch *MergePartialMatches(
  void *theEnv,
  struct partialMatch *list1,
  struct partialMatch *list2,
  int addActivationSlot,
  int addDependencySlot)
  {
   struct partialMatch *linker;
   short int endCount;
   static struct partialMatch mergeTemplate = { 1, 0, 0, 0, 0, 0, 0, 0}; /* betaMemory is TRUE */

   /*=================================*/
   /* Allocate the new partial match. */
   /*=================================*/
   
   endCount = addActivationSlot + addDependencySlot;
   linker = get_var_struct(theEnv,partialMatch,
                           sizeof(struct genericMatch) *
                            (list1->bcount + list2->bcount + endCount - 1));

   /*============================================*/
   /* Set the flags to their appropriate values. */
   /*============================================*/
   
   memcpy(linker,&mergeTemplate,sizeof(struct partialMatch) - sizeof(struct genericMatch));
   
   linker->activationf = addActivationSlot;
   linker->dependentsf = addDependencySlot;
   linker->bcount = list1->bcount + list2->bcount;
   
   /*========================================================*/
   /* Copy the bindings of the partial match being extended. */
   /*========================================================*/
      
   memcpy(linker->binds,list1->binds,sizeof(struct genericMatch) * list1->bcount);

   /*==============================================*/
   /* Add the bindings of the other partial match. */
   /*==============================================*/

   /* TBD: Should be only one, so this could be simplified. */
      
   memcpy(&linker->binds[list1->bcount],list2->binds,sizeof(struct genericMatch) * list2->bcount);

   /*==================================================*/
   /* The remainder of the structure should be nulled. */
   /*==================================================*/

   if (endCount)   
     { memset(&linker->binds[list1->bcount + list2->bcount],0,sizeof(struct genericMatch) * endCount); }

   return(linker);
  }

/*******************************************************************/
/* InitializePatternHeader: Initializes a pattern header structure */
/*   (used by the fact and instance pattern matchers).             */
/*******************************************************************/
globle void InitializePatternHeader(
  void *theEnv,
  struct patternNodeHeader *theHeader)
  {
#if MAC_MCW || IBM_MCW || MAC_XCD
#pragma unused(theEnv)
#endif
   theHeader->entryJoin = NULL;
   theHeader->rightHash = NULL;
   InitializeAlphaMemory(theHeader); /* 6.25 TBD Hashing */
   theHeader->singlefieldNode = FALSE;
   theHeader->multifieldNode = FALSE;
   theHeader->stopNode = FALSE;
#if (! RUN_TIME)
   theHeader->initialize = EnvGetIncrementalReset(theEnv);
#else
   theHeader->initialize = FALSE;
#endif
   theHeader->marked = FALSE;
   theHeader->beginSlot = FALSE;
   theHeader->endSlot = FALSE;
  }

/******************************************************************/
/* CreateAlphaMatch: Given a pointer to an entity (such as a fact */
/*   or instance) which matched a pattern, this function creates  */
/*   a partial match suitable for storing in the alpha memory of  */
/*   the pattern network. Note that the multifield markers which  */
/*   are passed as a calling argument are copied (thus the caller */
/*   is still responsible for freeing these data structures).     */
/******************************************************************/
globle struct partialMatch *CreateAlphaMatch(
  void *theEnv,
  void *theEntity,
  struct multifieldMarker *markers,
  struct patternNodeHeader *theHeader,
  unsigned long hashOffset)
  {
   struct partialMatch *theMatch;
   struct alphaMatch *afbtemp;
   unsigned long hashValue;
   struct alphaMemoryHash *theAlphaMemory;

   /*==================================================*/
   /* Create the alpha match and intialize its values. */
   /*==================================================*/

   theMatch = get_struct(theEnv,partialMatch);
   InitializePMLinks(theMatch);
   theMatch->betaMemory = FALSE;
   theMatch->busy = FALSE;
   theMatch->activationf = FALSE;
   theMatch->dependentsf = FALSE;
   theMatch->notOriginf = FALSE;
   theMatch->counterf = FALSE;
   theMatch->bcount = 1;
   theMatch->hashValue = hashOffset;
   theMatch->hashExpr = theHeader->rightHash;

   afbtemp = get_struct(theEnv,alphaMatch);
   afbtemp->next = NULL;
   afbtemp->matchingItem = (struct patternEntity *) theEntity;

   if (markers != NULL)
     { afbtemp->markers = CopyMultifieldMarkers(theEnv,markers); }
   else
     { afbtemp->markers = NULL; }

   theMatch->binds[0].gm.theMatch = afbtemp;

   /*============================================*/
   /* Find the alpha memory of the pattern node. */
   /*============================================*/

   hashValue = AlphaMemoryHashValue(theEnv,theHeader,hashOffset);
   theAlphaMemory = FindAlphaMemory(theEnv,theHeader,hashValue);
   afbtemp->bucket = hashValue;

   /*============================================*/
   /* Create an alpha memory if it wasn't found. */
   /*============================================*/

   if (theAlphaMemory == NULL)
     {
      theAlphaMemory = get_struct(theEnv,alphaMemoryHash);
      theAlphaMemory->bucket = hashValue;
      theAlphaMemory->owner = theHeader;
      theAlphaMemory->alphaMemory = NULL;
      theAlphaMemory->endOfQueue = NULL;
      theAlphaMemory->nextHash = NULL;

      theAlphaMemory->next = DefruleData(theEnv)->AlphaMemoryTable[hashValue];
      if (theAlphaMemory->next != NULL)
        { theAlphaMemory->next->prev = theAlphaMemory; }

      theAlphaMemory->prev = NULL; 
      DefruleData(theEnv)->AlphaMemoryTable[hashValue] = theAlphaMemory;
      
      if (theHeader->firstHash == NULL)
        {
         theHeader->firstHash = theAlphaMemory;
         theHeader->lastHash = theAlphaMemory;
         theAlphaMemory->prevHash = NULL;
        }
      else
        {
         theHeader->lastHash->nextHash = theAlphaMemory;
         theAlphaMemory->prevHash = theHeader->lastHash;
         theHeader->lastHash = theAlphaMemory;
        }
     }

   /*====================================*/
   /* Store the alpha match in the alpha */
   /* memory of the pattern node.        */
   /*====================================*/

 /* 6.25 TBD Hashing */
 
    theMatch->prevInNode = theAlphaMemory->endOfQueue;
    if (theAlphaMemory->endOfQueue == NULL)
     {
      theAlphaMemory->alphaMemory = theMatch;
      theAlphaMemory->endOfQueue = theMatch;
     }
   else
     {
      theAlphaMemory->endOfQueue->nextInNode = theMatch; /* TBD alphaBeta link - OK for now */
      theAlphaMemory->endOfQueue = theMatch;
     }

   /*===================================================*/
   /* Return a pointer to the newly create alpha match. */
   /*===================================================*/

   return(theMatch);
  }

/*********************************************************/
/* AddSingleMatch: Combines an alpha match and a partial */
/*   match into a new partial match.                     */
/*********************************************************/
globle struct partialMatch *AddSingleMatch(
  void *theEnv,
  struct partialMatch *list,
  struct alphaMatch *afb,
  int addActivationSlot,
  int addDependencySlot)
  {
   struct partialMatch *linker;
   short int i;

   linker = get_var_struct(theEnv,partialMatch,sizeof(struct genericMatch) *
                                        (list->bcount + addActivationSlot +
                                        addDependencySlot));
     
   memset(linker,0,sizeof(struct partialMatch));

   linker->betaMemory = TRUE;
   linker->activationf = addActivationSlot;
   linker->dependentsf = addDependencySlot;
   linker->bcount = list->bcount + 1;
   
   for (i = 0; i < (int) list->bcount; i++)
     { linker->binds[i] = list->binds[i]; }

   set_nth_pm_match(linker,i++,afb);

   if (addActivationSlot) linker->binds[i++].gm.theValue = NULL;
   if (addDependencySlot) linker->binds[i].gm.theValue = NULL;

   return(linker);
  }

/***********************************************************/
/* AddNotMatch: Creates an unsatisified partial match for  */
/*   the beta memory of a join that has negated RHS entry. */
/***********************************************************/
globle struct partialMatch *AddNotMatch(
  void *theEnv,
  struct partialMatch *list,
  int addActivationSlot,
  int addDependencySlot)
  {
   struct partialMatch *linker;
   short int endCount;
   static struct partialMatch notTemplate = { 1, 0, 0, 0, 1, 1, 0, 0}; /* betaMemory, notOriginf, and counterf are TRUE */

   /*=================================*/
   /* Allocate the new partial match. */
   /*=================================*/
   
   endCount = addActivationSlot + addDependencySlot;
   linker = get_var_struct(theEnv,partialMatch,sizeof(struct genericMatch) *
                                        (list->bcount + endCount));
   
   /*============================================*/
   /* Set the flags to their appropriate values. */
   /*============================================*/
   
   memcpy(linker,&notTemplate,sizeof(struct partialMatch) - sizeof(struct genericMatch));
   
   linker->activationf = addActivationSlot;
   linker->dependentsf = addDependencySlot;
   linker->bcount = list->bcount + 1;

   /*========================================================*/
   /* Copy the bindings of the partial match being extended. */
   /*========================================================*/
      
   memcpy(linker->binds,list->binds,sizeof(struct genericMatch) * list->bcount);
   
   /*==================================================*/
   /* The remainder of the structure should be nulled. */
   /*==================================================*/
   
   memset(&linker->binds[list->bcount],0,sizeof(struct genericMatch) * (endCount + 1));
   
   return(linker);
  }

/*******************************************/
/* CopyMultifieldMarkers: Copies a list of */
/*   multifieldMarker data structures.     */
/*******************************************/
struct multifieldMarker *CopyMultifieldMarkers(
  void *theEnv,
  struct multifieldMarker *theMarkers)
  {
   struct multifieldMarker *head = NULL, *lastMark = NULL, *newMark;

   while (theMarkers != NULL)
     {
      newMark = get_struct(theEnv,multifieldMarker);
      newMark->next = NULL;
      newMark->whichField = theMarkers->whichField;
      newMark->where = theMarkers->where;
      newMark->startPosition = theMarkers->startPosition;
      newMark->endPosition = theMarkers->endPosition;

      if (lastMark == NULL)
        { head = newMark; }
      else
        { lastMark->next = newMark; }
      lastMark = newMark;

      theMarkers = theMarkers->next;
     }

   return(head);
  }

/***************************************************************/
/* NewPseudoFactPartialMatch: Creates a partial structure that */
/*   indicates the "pseudo" fact to which a not pattern CE has */
/*   been bound. Since a non-existant fact has no fact index,  */
/*   the partial match structure is given a pseudo fact index  */
/*   (a unique negative integer). Note that a "pseudo" fact    */
/*   can also be used as a "pseudo" instance.                  */
/***************************************************************/
globle struct partialMatch *NewPseudoFactPartialMatch(
  void *theEnv)
  {
   struct partialMatch *linker;
   struct alphaMatch *tempAlpha;

   linker = get_struct(theEnv,partialMatch);
   InitializePMLinks(linker);
   linker->betaMemory = TRUE;
   linker->busy = FALSE;
   linker->activationf = FALSE;
   linker->dependentsf = FALSE;
   linker->notOriginf = TRUE;
   linker->counterf = FALSE;
   /* linker->invalid = FALSE; */
   linker->bcount = 0;
   tempAlpha = get_struct(theEnv,alphaMatch);
   tempAlpha->next = NULL;
   tempAlpha->matchingItem = NULL;
   tempAlpha->markers = NULL;

   linker->binds[0].gm.theMatch = tempAlpha;
   return(linker);
  }

/******************************************************************/
/* FlushAlphaBetaMemory: Returns all partial matches in a list of */
/*   partial matches either directly to the pool of free memory   */
/*   or to the list of GarbagePartialMatches. Partial matches     */
/*   stored in alpha memories and partial matches which store the */
/*   information for pseudo facts (for not CEs) may be referred   */
/*   to by other data structures and thus must be placed on the   */
/*   list of GarbagePartialMatches.                               */
/******************************************************************/
globle void FlushAlphaBetaMemory(
  void *theEnv,
  struct partialMatch *pfl)
  {
   struct partialMatch *pfltemp;

   while (pfl != NULL)
     {
      pfltemp = pfl->nextInNode;

      UnlinkBetaPartialMatchfromAlphaAndBetaLineage(pfl);

      if (((pfl->notOriginf) && (pfl->counterf == FALSE)) ||
          (pfl->betaMemory == FALSE))
        {
         pfl->nextInNode = EngineData(theEnv)->GarbagePartialMatches; /* TBD alphaBeta links - Done */
         EngineData(theEnv)->GarbagePartialMatches = pfl;
        }
      else
        { ReturnPartialMatch(theEnv,pfl); }

      pfl = pfltemp;
     }
  }

/*****************************************************************/
/* DestroyAlphaBetaMemory: Returns all partial matches in a list */
/*   of partial matches directly to the pool of free memory.     */
/*****************************************************************/
globle void DestroyAlphaBetaMemory(
  void *theEnv,
  struct partialMatch *pfl)
  {
   struct partialMatch *pfltemp;

   while (pfl != NULL)
     {
      pfltemp = pfl->nextInNode;
      DestroyPartialMatch(theEnv,pfl); 
      pfl = pfltemp;
     }
  }

/******************************************************/
/* FindEntityInPartialMatch: Searches for a specified */
/*   data entity in a partial match.                  */
/******************************************************/
globle int FindEntityInPartialMatch(
  struct patternEntity *theEntity,
  struct partialMatch *thePartialMatch)
  {
   short int i;

   for (i = 0 ; i < (int) thePartialMatch->bcount; i++)
     {
      if (thePartialMatch->binds[i].gm.theMatch == NULL) continue;
      if (thePartialMatch->binds[i].gm.theMatch->matchingItem == theEntity)
        { return(TRUE); }
     }

   return(FALSE);
  }

/******************************************************/
/* PartialMatchInvalid:                  */
/******************************************************/
globle int PartialMatchInvalid(
  struct partialMatch *thePartialMatch)
  {
  /*
   short int i;

   for (i = 0 ; i < (int) thePartialMatch->bcount; i++)
     {
      if (thePartialMatch->binds[i].gm.theMatch != NULL)
        { 
         if (thePartialMatch->binds[i].gm.theMatch->invalid)
           { return(TRUE); }
        }
     }
*/
   return(FALSE);
  }
  
/***********************************************************************/
/* GetPatternNumberFromJoin: Given a pointer to a join associated with */
/*   a pattern CE, returns an integer representing the position of the */
/*   pattern CE in the rule (e.g. first, second, third).               */
/***********************************************************************/
globle int GetPatternNumberFromJoin(
  struct joinNode *joinPtr)
  {
   int whichOne = 0;

   while (joinPtr != NULL)
     {
      if (joinPtr->joinFromTheRight)
        { joinPtr = (struct joinNode *) joinPtr->rightSideEntryStructure; }
      else
        {
         whichOne++;
         joinPtr = joinPtr->lastLevel;
        }
     }

   return(whichOne);
  }

/************************************************************************/
/* TraceErrorToRule: Prints an error message when a error occurs as the */
/*   result of evaluating an expression in the pattern network. Used to */
/*   indicate which rule caused the problem.                            */
/************************************************************************/
globle void TraceErrorToRule(
  void *theEnv,
  struct joinNode *joinPtr,
  char *indentSpaces)
  {
   MarkRuleNetwork(theEnv,0);
   TraceErrorToRuleDriver(theEnv,joinPtr,indentSpaces);
  }

/**************************************************************/
/* TraceErrorToRuleDriver: Driver code for printing out which */
/*   rule caused a pattern or join network error.             */
/**************************************************************/
static void TraceErrorToRuleDriver(
  void *theEnv,
  struct joinNode *joinPtr,
  char *indentSpaces)
  {
   char *name;

   while (joinPtr != NULL)
     {
      if (joinPtr->marked)
        { /* Do Nothing */ }
      else if (joinPtr->ruleToActivate != NULL)
        {
         joinPtr->marked = 1;
         name = EnvGetDefruleName(theEnv,joinPtr->ruleToActivate);
         EnvPrintRouter(theEnv,WERROR,indentSpaces);
         EnvPrintRouter(theEnv,WERROR,name);
         EnvPrintRouter(theEnv,WERROR,"\n");
        }
      else
        {
         joinPtr->marked = 1;
         TraceErrorToRuleDriver(theEnv,joinPtr->nextLevel,indentSpaces);
        }

      joinPtr = joinPtr->rightDriveNode;
     }
  }

/********************************************************/
/* MarkRuleNetwork: Sets the marked flag in each of the */
/*   joins in the join network to the specified value.  */
/********************************************************/
globle void MarkRuleNetwork(
  void *theEnv,
  int value)
  {
   struct defrule *rulePtr;
   struct joinNode *joinPtr;
   struct defmodule *modulePtr;

   /*===========================*/
   /* Loop through each module. */
   /*===========================*/

   SaveCurrentModule(theEnv);
   for (modulePtr = (struct defmodule *) EnvGetNextDefmodule(theEnv,NULL);
        modulePtr != NULL;
        modulePtr = (struct defmodule *) EnvGetNextDefmodule(theEnv,modulePtr))
     {
      EnvSetCurrentModule(theEnv,(void *) modulePtr);

      /*=========================*/
      /* Loop through each rule. */
      /*=========================*/

      rulePtr = (struct defrule *) EnvGetNextDefrule(theEnv,NULL);
      while (rulePtr != NULL)
        {
         /*=============================*/
         /* Mark each join for the rule */
         /* with the specified value.   */
         /*=============================*/

         joinPtr = rulePtr->lastJoin;
         while (joinPtr != NULL)
           {
            joinPtr->marked = value;
            joinPtr = GetPreviousJoin(joinPtr);
           }

         /*=================================*/
         /* Move on to the next rule or the */
         /* next disjunct for this rule.    */
         /*=================================*/

         if (rulePtr->disjunct != NULL) rulePtr = rulePtr->disjunct;
         else rulePtr = (struct defrule *) EnvGetNextDefrule(theEnv,rulePtr);
        }

     }

   RestoreCurrentModule(theEnv);
  }

/*************************************************/
/* GetAlphaMemory: Retrieves the list of matches */
/*   matches from an alpha memory.               */
/*************************************************/ 
globle struct partialMatch *GetAlphaMemory(
  void *theEnv,
  struct patternNodeHeader *theHeader,
  unsigned long hashOffset)
  {
   struct alphaMemoryHash *theAlphaMemory;
   unsigned long hashValue;

   hashValue = AlphaMemoryHashValue(theEnv,theHeader,hashOffset);
   theAlphaMemory = FindAlphaMemory(theEnv,theHeader,hashValue);

   if (theAlphaMemory == NULL)
     { return NULL; }

   return theAlphaMemory->alphaMemory;
  }

/*********************************************/
/* RemoveAlphaMemoryMatches: Removes matches */
/*   from an alpha memory.                   */
/*********************************************/ 
globle void RemoveAlphaMemoryMatches(
  void *theEnv,
  struct patternNodeHeader *theHeader,
  struct partialMatch *theMatch,
  struct alphaMatch *theAlphaMatch,
  unsigned long hashOffset)
  {
   struct alphaMemoryHash *theAlphaMemory = NULL;
   unsigned long hashValue;

   if ((theMatch->prevInNode == NULL) || (theMatch->nextInNode == NULL)) /* TBD alphaBeta links */
     {
      hashValue = theAlphaMatch->bucket;
      theAlphaMemory = FindAlphaMemory(theEnv,theHeader,hashValue);
     }
     
   if (theMatch->prevInNode != NULL)
     { theMatch->prevInNode->nextInNode = theMatch->nextInNode; }
   else
     { theAlphaMemory->alphaMemory = theMatch->nextInNode; }

   if (theMatch->nextInNode != NULL)
     { theMatch->nextInNode->prevInNode = theMatch->prevInNode; }
   else
     { theAlphaMemory->endOfQueue = theMatch->prevInNode; }

    /* TBD -- links through the alpha/beta should already be removed */
    /* so no need to update these links at this point. */
   /*====================================*/
   /* Add the match to the garbage list. */
   /*====================================*/

   theMatch->nextInNode = EngineData(theEnv)->GarbagePartialMatches;
   EngineData(theEnv)->GarbagePartialMatches = theMatch;

   if ((theAlphaMemory != NULL) && (theAlphaMemory->alphaMemory == NULL))
     { UnlinkAlphaMemory(theEnv,theHeader,theAlphaMemory); }
  }

/*****************************************************************/
/* DestroyAlphaMemory: TBD added for hashed alpha memory */
/*****************************************************************/
globle void DestroyAlphaMemory(
  void *theEnv,
  struct patternNodeHeader *theHeader,
  int unlink)
  {
   struct alphaMemoryHash *theAlphaMemory, *tempMemory;

   theAlphaMemory = theHeader->firstHash;

   while (theAlphaMemory != NULL)
     {
      tempMemory = theAlphaMemory->nextHash;
      DestroyAlphaBetaMemory(theEnv,theAlphaMemory->alphaMemory); 
      if (unlink)
        { UnlinkAlphaMemoryBucketSiblings(theEnv,theAlphaMemory); }
      rtn_struct(theEnv,alphaMemoryHash,theAlphaMemory);
      theAlphaMemory = tempMemory;
     }

   theHeader->firstHash = NULL;
   theHeader->lastHash = NULL;
  }

/*****************************************************************/
/* FlushAlphaMemory: TBD added for hashed alpha memory */
/*****************************************************************/
globle void FlushAlphaMemory(
  void *theEnv,
  struct patternNodeHeader *theHeader)
  {
   struct alphaMemoryHash *theAlphaMemory, *tempMemory;

   theAlphaMemory = theHeader->firstHash;

   while (theAlphaMemory != NULL)
     {
      tempMemory = theAlphaMemory->nextHash;
      FlushAlphaBetaMemory(theEnv,theAlphaMemory->alphaMemory); 
      UnlinkAlphaMemoryBucketSiblings(theEnv,theAlphaMemory);
      rtn_struct(theEnv,alphaMemoryHash,theAlphaMemory);
      theAlphaMemory = tempMemory;
     }

   theHeader->firstHash = NULL;
   theHeader->lastHash = NULL;
  }

/*****************************************************************/
/* FindAlphaMemory:  */
/*****************************************************************/
static struct alphaMemoryHash *FindAlphaMemory(
  void *theEnv,
  struct patternNodeHeader *theHeader,
  unsigned long hashValue)
  {
   struct alphaMemoryHash *theAlphaMemory;
      
   theAlphaMemory = DefruleData(theEnv)->AlphaMemoryTable[hashValue];

   if (theAlphaMemory != NULL)
     {
      while ((theAlphaMemory != NULL) && (theAlphaMemory->owner != theHeader))
        { theAlphaMemory = theAlphaMemory->next; }
     }
     
   return theAlphaMemory;
  }   

/*****************************************************************/
/* AlphaMemoryHashValue:  */
/*****************************************************************/
static unsigned long AlphaMemoryHashValue(
  void *theEnv,
  struct patternNodeHeader *theHeader,
  unsigned long hashOffset)
  {
   unsigned long hashValue;
   
   hashValue = (unsigned long) theHeader + hashOffset;
   hashValue = hashValue % ALPHA_MEMORY_HASH_SIZE;
   
   return hashValue;
  }
  
/*****************************************************************/
/* UnlinkAlphaMemory:  */
/*****************************************************************/
static void UnlinkAlphaMemory(
  void *theEnv,
  struct patternNodeHeader *theHeader,
  struct alphaMemoryHash *theAlphaMemory)
  {
   /*======================*/
   /* Unlink the siblings. */
   /*======================*/
    
   UnlinkAlphaMemoryBucketSiblings(theEnv,theAlphaMemory);
      
   /*================================*/
   /* Update firstHash and lastHash. */
   /*================================*/
   
   if (theAlphaMemory == theHeader->firstHash)
     { theHeader->firstHash = theAlphaMemory->nextHash; }
     
   if (theAlphaMemory == theHeader->lastHash)
     { theHeader->lastHash = theAlphaMemory->prevHash; }
        
   /*===============================*/
   /* Update nextHash and prevHash. */
   /*===============================*/

   if (theAlphaMemory->prevHash != NULL)
     { theAlphaMemory->prevHash->nextHash = theAlphaMemory->nextHash; }
     
   if (theAlphaMemory->nextHash != NULL)
     { theAlphaMemory->nextHash->prevHash = theAlphaMemory->prevHash; }
              
   rtn_struct(theEnv,alphaMemoryHash,theAlphaMemory);
  }   

/*****************************************************************/
/* UnlinkAlphaMemoryBucketSiblings:  */
/*****************************************************************/
static void UnlinkAlphaMemoryBucketSiblings(
  void *theEnv,
  struct alphaMemoryHash *theAlphaMemory)
  {
   if (theAlphaMemory->prev == NULL)
     { DefruleData(theEnv)->AlphaMemoryTable[theAlphaMemory->bucket] = theAlphaMemory->next; }
   else
     { theAlphaMemory->prev->next = theAlphaMemory->next; }
        
   if (theAlphaMemory->next != NULL)
     { theAlphaMemory->next->prev = theAlphaMemory->prev; }
  }   

/********************************************/
/* InitializeAlphaMememory: Initializes the */
/*    alpha memory of a pattern node.       */
/********************************************/ 
globle void InitializeAlphaMemory(
  struct patternNodeHeader *theHeader)
  {
   theHeader->firstHash = NULL;
   theHeader->lastHash = NULL;
   //theHeader->xalphaMemory = NULL;
   //theHeader->xendOfQueue = NULL;
  }

/********************************************/
/* ComputeRightHashValue:       */
/********************************************/ 
unsigned long ComputeRightHashValue(
  void *theEnv,
  struct patternNodeHeader *theHeader)
  {
   struct expr *tempExpr;
   unsigned long hashValue = 0;
   unsigned long multiplier = 1;
      
   if (theHeader->rightHash == NULL)
     { return hashValue; }
     
   for (tempExpr = theHeader->rightHash; 
        tempExpr != NULL; 
        tempExpr = tempExpr->nextArg, multiplier = multiplier * 509)
      {
       DATA_OBJECT theResult;
       struct expr *oldArgument;
        
       oldArgument = EvaluationData(theEnv)->CurrentExpression;
       EvaluationData(theEnv)->CurrentExpression = tempExpr;
       (*EvaluationData(theEnv)->PrimitivesArray[tempExpr->type]->evaluateFunction)(theEnv,tempExpr->value,&theResult);
       EvaluationData(theEnv)->CurrentExpression = oldArgument;
        
       switch (theResult.type)
         {
          case STRING:
          case SYMBOL:
          case INSTANCE_NAME:
            hashValue += (((SYMBOL_HN *) theResult.value)->bucket * multiplier);
            break;
             
          case INTEGER:
            hashValue += (((INTEGER_HN *) theResult.value)->bucket * multiplier);
            break;
             
          case FLOAT:
            hashValue += (((FLOAT_HN *) theResult.value)->bucket * multiplier);
            break;
          }
       }
       
     return hashValue;
    }

#if (CONSTRUCT_COMPILER || BLOAD_AND_BSAVE) && (! RUN_TIME)

/*************************************************************/
/* TagRuleNetwork: Assigns each join in the join network and */
/*   each defrule data structure with a unique integer ID.   */
/*   Also counts the number of defrule and joinNode data     */
/*   structures currently in use.                            */
/*************************************************************/
globle void TagRuleNetwork(
  void *theEnv,
  long int *moduleCount,
  long int *ruleCount,
  long int *joinCount)
  {
   struct defmodule *modulePtr;
   struct defrule *rulePtr;
   struct joinNode *joinPtr;

   *moduleCount = 0;
   *ruleCount = 0;
   *joinCount = 0;

   MarkRuleNetwork(theEnv,0);

   /*===========================*/
   /* Loop through each module. */
   /*===========================*/

   for (modulePtr = (struct defmodule *) EnvGetNextDefmodule(theEnv,NULL);
        modulePtr != NULL;
        modulePtr = (struct defmodule *) EnvGetNextDefmodule(theEnv,modulePtr))
     {
      (*moduleCount)++;
      EnvSetCurrentModule(theEnv,(void *) modulePtr);

      /*=========================*/
      /* Loop through each rule. */
      /*=========================*/

      rulePtr = (struct defrule *) EnvGetNextDefrule(theEnv,NULL);

      while (rulePtr != NULL)
        {
         rulePtr->header.bsaveID = *ruleCount;
         (*ruleCount)++;

         /*=========================*/
         /* Loop through each join. */
         /*=========================*/

         for (joinPtr = rulePtr->lastJoin;
              joinPtr != NULL;
              joinPtr = GetPreviousJoin(joinPtr))
           {
            if (joinPtr->marked == 0)
              {
               joinPtr->marked = 1;
               joinPtr->bsaveID = *joinCount;
               (*joinCount)++;
              }
           }

         if (rulePtr->disjunct != NULL) rulePtr = rulePtr->disjunct;
         else rulePtr = (struct defrule *) EnvGetNextDefrule(theEnv,rulePtr);
        }
     }
  }

#endif /* (CONSTRUCT_COMPILER || BLOAD_AND_BSAVE) && (! RUN_TIME) */

#endif /* DEFRULE_CONSTRUCT */





